'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var arraybufferXmlParser = require('arraybuffer-xml-parser');
var pako = require('pako');
var uint8Base64 = require('uint8-base64');
var camelCase = require('camelcase');

function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e : { 'default': e }; }

var pako__default = /*#__PURE__*/_interopDefaultLegacy(pako);
var camelCase__default = /*#__PURE__*/_interopDefaultLegacy(camelCase);

function decodeBase64(base64, options = {}) {
  let {
    endian = 'little',
    precision,
    float = true,
    compression = '',
    ontologies,
  } = options;

  if (ontologies) {
    if (ontologies.includes('MS:1000519')) {
      precision = 32;
      float = false;
    }
    if (ontologies.includes('MS:1000520')) precision = 16;
    if (ontologies.includes('MS:1000521')) precision = 32;
    if (ontologies.includes('MS:1000522')) {
      float = false;
      precision = 64;
    }
    if (ontologies.includes('MS:1000523')) precision = 64;
    if (ontologies.includes('MS:1000574')) compression = 'zlib';
  }

  let uint8Array = uint8Base64.decode(base64);
  switch (compression.toLowerCase()) {
    case 'zlib':
      uint8Array = pako__default['default'].inflate(uint8Array);
      break;
    case '':
    case 'none':
      break;
    default:
      throw new Error(`Unknow compression algorithm: ${compression}`);
  }

  switch (endian.toLowerCase()) {
    case 'little':
      break;
    case 'network':
    case 'big':
      {
        // we will invert in place the data
        let step;
        switch (precision) {
          case 32:
            step = 4;
            break;
          case 64:
            step = 8;
            break;
          default:
            throw new Error('Can not process bigendian file');
        }
        for (
          let i = 0;
          i < uint8Array.length - (uint8Array.length % step);
          i += step
        ) {
          for (let j = 0; j < step / 2; j++) {
            const temp = uint8Array[i + j];
            uint8Array[i + j] = uint8Array[i + step - 1 - j];
            uint8Array[i + step - 1 - j] = temp;
          }
        }
      }
      break;
    default:
      throw new TypeError(`Attributes endian not correct: ${endian}`);
  }

  /*
       We should take care that the length of the Uint8Array is correct but the buffer
       may be a little bit bigger because when decoding base 64 it may end with = or ==
       and we plan the size in the buffer.
      */
  if (float) {
    switch (precision) {
      case 32:
        return new Float32Array(uint8Array.buffer, 0, uint8Array.length / 4);
      case 64:
        return new Float64Array(uint8Array.buffer, 0, uint8Array.length / 8);
      default:
        throw new TypeError(`Incorrect precision: ${precision}`);
    }
  } else {
    switch (precision) {
      case 32:
        return new Int32Array(uint8Array.buffer, 0, uint8Array.length / 4);
      case 64:
        // eslint-disable-next-line no-undef
        return new BigInt64Array(uint8Array.buffer, 0, uint8Array.length / 8);
      default:
        throw new TypeError(`Incorrect precision: ${precision}`);
    }
  }
}

function parseCvParam$1(cvParam) {
  let result = {};
  if (!cvParam) return result;
  let cvParams;
  if (Array.isArray(cvParam)) {
    cvParams = cvParam;
  } else {
    cvParams = [cvParam];
  }
  for (let param of cvParams) {
    let attributes = param.attributes;
    if (attributes.name) {
      result[attributes.name.toLowerCase()] = {
        accession: attributes.accession,
        cvLabel: attributes.cvLabel,
        value: attributes.value,
        name: attributes.name,
      };
    }
  }
  return result;
}

function processMetadata(parsed, metadata) {
  if (!parsed || !parsed.description) return;
  let description = parsed.description;
  if (description.dataProcessing) {
    let dataProcessing = description.dataProcessing;
    if (dataProcessing.software && dataProcessing.software.name) {
      metadata.software = dataProcessing.software.name;
    }
  }
  if (description.instrument) {
    let instrument = description.instrument;
    if (instrument.analyzerList && instrument.analyzerList.analyzer) {
      let analyzer = instrument.analyzerList.analyzer;
      let cvParam = parseCvParam$1(analyzer.cvParam);
      if (cvParam.analyzertype) {
        metadata.analyzer = cvParam.analyzertype.value;
      }
    }
    if (instrument.detector) {
      let detector = instrument.detector;
      let cvParam = parseCvParam$1(detector.cvParam);
      if (cvParam.detectortype) {
        metadata.detector = cvParam.detectortype.value;
      }
    }
  }
}

function processSpectrumList$2(parsed, times, msData) {
  if (!parsed || !parsed.spectrumList || !parsed.spectrumList.spectrum) return;
  let spectrumList = parsed.spectrumList.spectrum;
  for (let spectrum of spectrumList) {
    let info = parseCvParam$1(
      spectrum.spectrumDesc.spectrumSettings.spectrumInstrument.cvParam,
    );

    times.push(info.timeinminutes.value);

    let mzArray = spectrum.mzArrayBinary.data['#text'] || [];
    let intensity = spectrum.intenArrayBinary.data['#text'] || [];
    msData.push([mzArray, intensity]);
  }
}

const decoder$3 = new TextDecoder();

function parseMzData(arrayBuffer) {
  const result = {
    metadata: {},
    times: [],
    series: {
      ms: {
        data: [],
      },
    },
  };

  let parsed = arraybufferXmlParser.parse(arrayBuffer, {
    attributeNamePrefix: '',
    attributesNodeName: 'attributes',
    tagValueProcessor: (value, node) => {
      if (node.tagName !== 'data') return decoder$3.decode(value);
      return decodeBase64(node.value, node.attributes);
    },
  });

  processMetadata(parsed.mzData, result.metadata);
  processSpectrumList$2(parsed.mzData, result.times, result.series.ms.data);

  return result;
}

function parseCvParam(cvParam) {
  let result = {};
  if (!cvParam) return result;
  let cvParams;
  if (Array.isArray(cvParam)) {
    cvParams = cvParam;
  } else {
    cvParams = [cvParam];
  }
  for (let parameter of cvParams) {
    let attribute = parameter.attributes;
    if (attribute.name) {
      result[
        camelCase__default['default'](attribute.name.toLowerCase().replace(/[^ a-z0-9]/g, ''))
      ] = {
        accession: attribute.accession,
        cvLabel: attribute.cvLabel,
        value: attribute.value,
        name: attribute.name,
      };
    }
  }
  return result;
}

function processSpectrumList$1(parsed, times, msData) {
  if (
    !parsed ||
    !parsed.run ||
    !parsed.run.spectrumList ||
    !parsed.run.spectrumList.spectrum
  ) {
    return;
  }
  let spectrumList = parsed.run.spectrumList.spectrum;

  for (let spectrum of spectrumList) {
    if (!spectrum.binaryDataArrayList) continue;
    let scanList = spectrum.scanList;
    if (Array.isArray(scanList)) throw new Error('Unsupported scanList');

    let scan = scanList.scan;

    if (typeof scan !== 'object') continue;
    if (Array.isArray(scan)) {
      throw new Error('processSpectrumList: scan may not be an array');
    }
    const cvParam = parseCvParam(scan.cvParam);
    times.push(cvParam.scanStartTime.value);

    const dataArrayList = spectrum.binaryDataArrayList.binaryDataArray;
    if (dataArrayList.length !== 2) {
      throw new Error('Can not decodeData because length !== 2');
    }

    const first = dataArrayList[0];
    const firstCVParams = parseCvParam(first.cvParam);
    const second = dataArrayList[1];
    const secondCVParams = parseCvParam(second.cvParam);

    if (firstCVParams.mzArray && secondCVParams.intensityArray) {
      msData.push([first.binary, second.binary]);
    }
    if (firstCVParams.intensityArray && secondCVParams.mzArray) {
      msData.push([second.binary, first.binary]);
    }
  }
}

const decoder$2 = new TextDecoder();

function parseMzML(arrayBuffer) {
  const result = {
    metadata: {},
    times: [],
    series: {
      ms: {
        data: [],
      },
    },
  };

  let parsed = arraybufferXmlParser.parse(arrayBuffer, {
    attributeNamePrefix: '',
    attributesNodeName: 'attributes',
    tagValueProcessor: (value, node) => {
      if (node.tagName !== 'binary') return decoder$2.decode(value);
      const ontologies = node.parent.children.cvParam.map(
        (entry) => entry.attributes.accession,
      );

      return decodeBase64(node.value, { ontologies });
    },
  });

  const mzML = parsed.mzML || parsed.indexedmzML.mzML;

  processSpectrumList$1(mzML, result.times, result.series.ms.data);

  return result;
}

function processSpectrumList(parsed, times, msData) {
  if (!parsed.msRun.scan) return;
  let scanList = parsed.msRun.scan;
  if (Array.isArray(scanList) === false) scanList = [scanList];
  if (scanList[0].attributes) msData.info = [];
  for (let scan of scanList) {
    if (typeof scan !== 'object') continue;
    if (Array.isArray(scan)) {
      throw new Error('processSpectrumList: scan may not be an array');
    }
    const dataArray = scan.peaks['#text'];
    let length = dataArray.length / 2;
    let first = new Float64Array(length);
    let second = new Float64Array(length);
    for (let i = 0; i < length; i++) {
      first[i] = dataArray[i * 2];
      second[i] = dataArray[i * 2 + 1];
    }
    msData.data.push([first, second]);
    msData.info.push(scan.attributes);
    times.push(
      parseFloat(
        scan.attributes.retentionTime.replace(/(?:P*)(?:T*)(?:S*)/gi, ''),
      ),
    );
  }
}

const decoder$1 = new TextDecoder();

function parseMzXML(arrayBuffer) {
  const result = {
    metadata: {},
    times: [],
    series: {
      ms: {
        data: [],
      },
    },
  };
  let parsed = arraybufferXmlParser.parse(arrayBuffer, {
    attributeNamePrefix: '',
    attributesNodeName: 'attributes',
    tagValueProcessor: (value, node) => {
      if (node.tagName !== 'peaks') return decoder$1.decode(value);
      return decodeBase64(node.value, {
        precision: node.attributes.precision,
        endian: node.attributes.byteOrder,
        compression: node.attributes.compressionType,
      });
    },
  });

  processSpectrumList(parsed.mzXML, result.times, result.series.ms);

  return result;
}

const decoder = new TextDecoder();

/**
 * Reads a mzData v1.05 file
 * @param {ArrayBuffer|string} xml - ArrayBuffer or String or any Typed Array (including Node.js' Buffer from v4) with the data
 * @return {{times: Array<number>, series: { ms: { data:Array<Array<number>>}}}}
 */
function parseMZ(xml) {
  if (typeof xml === 'string') {
    const encoder = new TextEncoder();
    xml = encoder.encode(xml);
  }

  if (!ArrayBuffer.isView(xml)) {
    xml = new Uint8Array(xml);
  }

  const header = xml.subarray
    ? decoder.decode(xml.subarray(0, 200))
    : xml.substring(0, 200);

  if (header.includes('mzData')) {
    return parseMzData(xml);
  } else if (header.includes('mzML')) {
    return parseMzML(xml);
  } else if (header.includes('mzXML')) {
    return parseMzXML(xml);
  } else {
    throw new Error(`MZ parser: unknown format`);
  }
}

exports.parseMZ = parseMZ;
